<?php
/* --------------------------------------------------------------
   ParcelServiceDescriptions.inc.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService\Model;

use ArrayIterator;
use Gambio\Core\Language\LanguageId;
use IteratorAggregate;
use JsonSerializable;
use Traversable;
use Webmozart\Assert\Assert;

/**
 * Class ParcelServiceDescriptions
 *
 * @package Gambio\Admin\ParcelService\Model
 */
class ParcelServiceDescriptions implements IteratorAggregate, JsonSerializable
{
    /**
     * @var ParcelServiceDescription[]
     */
    private $descriptions = [];
    
    
    /**
     * @return ParcelServiceDescriptions
     */
    public static function create(): ParcelServiceDescriptions
    {
        return new self();
    }
    
    
    /**
     * @param ParcelServiceDescription[] $descriptions
     *
     * @return ParcelServiceDescriptions
     */
    public static function createFromArray(array $descriptions = []): ParcelServiceDescriptions
    {
        Assert::allKeyExists($descriptions, 'languageId', 'All description items need a "languageId" key.');
        Assert::allKeyExists($descriptions, 'url', 'All description items need a "url" key.');
        Assert::allKeyExists($descriptions, 'comment', 'All description items need a "comment" key.');
        
        $collection = new self();
        foreach ($descriptions as $description) {
            $collection->add($description['languageId'],
                             ParcelServiceDescription::create($description['url'], $description['comment']));
        }
        
        return $collection;
    }
    
    
    /**
     * @param int                      $languageId
     * @param ParcelServiceDescription $description
     *
     * @return $this
     */
    public function add(int $languageId, ParcelServiceDescription $description): self
    {
        Assert::greaterThan($languageId, 0, 'Language ID must be greater than 0. Got: %s');
        
        $this->descriptions[$languageId] = $description;
        
        return $this;
    }
    
    
    /**
     * @param LanguageId $languageId
     *
     * @return string
     */
    public function url(LanguageId $languageId): string
    {
        if (array_key_exists($languageId->value(), $this->descriptions) === false) {
            return '';
        }
        
        return $this->descriptions[$languageId->value()]->url();
    }
    
    
    /**
     * @param LanguageId $languageId
     *
     * @return string
     */
    public function comment(LanguageId $languageId): string
    {
        if (array_key_exists($languageId->value(), $this->descriptions) === false) {
            return '';
        }
        
        return $this->descriptions[$languageId->value()]->comment();
    }
    
    
    /**
     * @inheritDoc
     */
    public function getIterator(): Traversable
    {
        return new ArrayIterator($this->descriptions);
    }
    
    
    /**
     * @return array
     *
     */
    public function toArray(): array
    {
        $descriptions = [];
        
        /**
         * @var int                      $languageId
         * @var ParcelServiceDescription $description
         */
        foreach ($this->descriptions as $languageId => $description) {
            $descriptions[] = array_merge($description->toArray(), ['languageId' => $languageId]);
        }
        
        return $descriptions;
    }
    
    
    /**
     * @inheritDoc
     */
    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
}

